/*
 * A simple interface to loading the fabric where the caller doesn't
 * have to know about database specific structures
 */

#include "db.h"

#include "libfma.h"
#include "lf_internal.h"
#include "lf_db.h"
#include "lf_fabric.h"
#include "lf_load_fabric.h"

/*
 * Local vars
 */
static char *FMS_run;
static char *DB_name;

/*
 * Set the FMS run and database name
 */
void
lf_simple_set_fabric_dirs(
  char *run,
  char *db_name)
{
  if (run != NULL) {
    FMS_run = run;
  }

  if (db_name != NULL) {
    DB_name = db_name;
  }
}

/*
 * Set the FMS run and DB directory if needed, the
 * load the fabric.
 */
lf_fabric_t *
lf_simple_load_fabric()
{
  struct lf_fabric_db *fdp;
  db_database_ptr_t dbp;
  lf_fabric_t *fp;

  /*
   * If run is not set, try to first get it from environment
   * and use default value if that does not work.  Then do the same
   * for the database name
   */
  if (FMS_run == NULL) FMS_run = getenv(LF_ENV_FMS_RUN);
  if (FMS_run == NULL || *FMS_run == '\0') FMS_run = LF_DFLT_FMS_RUN;

  if (DB_name == NULL) DB_name = getenv(LF_ENV_DB_NAME);
  if (DB_name == NULL || *DB_name == '\0') DB_name = LF_DFLT_DB_NAME;

  /*
   * Open the database associated with this fabric (readonly)
   */
  dbp = db_open_database(FMS_run, DB_name, FALSE);
  if (dbp == NULL) LF_ERROR(("Error opening database"));

  /*
   * now read the fabric
   */
  fdp = lf_fabric_db_handle(dbp);
  if (fdp == NULL) LF_ERROR(("Error getting fabric DB handle"));

  fp = lf_load_fabric(fdp);
  if (fp == NULL) {
    LF_ERROR(("Error reading fabric, run=%s, db=%s", FMS_run, DB_name));
  }

  /* close up the database tables */
  lf_close_fabric_db(fdp);
  LF_FREE(fdp);

  return fp;

 except:
  return NULL;
}
